package loquebot.drives;

import java.util.Iterator;
import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.Item;

import loquebot.Main;
import loquebot.memory.LoqueMemory;
import loquebot.body.LoqueTravel;
import loquebot.body.LoqueNavigator;

/**
 * Responsible for searching for an enemy. This drive handles wandering around
 * items (useful if possible) and navigation points in order to find the enemy.
 *
 * <h4>Prioritize healing</h4>
 *
 * Although the {@link LoquePowerUp} drive is responsible for powering up, it
 * also creates space for {@link LoquePursue} to make its choice after battles.
 * In order to do that (e.g. pursue wounded enemies even when health is less
 * than 50), the power-up drive does not force the health level to max.
 *
 * <p>This is done here as the first wandering thing: heal and get some armor.
 * While healing, also get some udamage on the way.</p>
 *
 * <h4>Further foraging</h4>
 *
 * First the drive tries to get items the agent could use somehow. Then, as the
 * agent gets stuffed, wanders around all pickable items (weapons, vials, armor,
 * udamage, adrenaline, etc.).
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueWander extends LoqueWanderBase
{
    /**
     * Id of the current travel ticket.
     */
    protected int travelTicket = -1;

    /*========================================================================*/

    /**
     * Main logic of the power-up. Drives the choosing of good useful items and
     * picking them up.
     *
     * <h4>Cook book</h4>
     * <ul>
     * <li>Check, whether we have items to travel to. Travel to those items but
     * also try to keep looking for possible dangers that might come.</li>
     * <li>Upon no items chosen, check, whether we shall not continue some
     * power-up driving. Although, {@link LoquePowerUp} is mightly responsible
     * for powering up, that drive also needs to think about {@link LoquePursue}
     * execution in appropriate situations, therefore {@link LoquePowerUp} does
     * not fully heal the agent and also might leave armor and udamage out.
     * Therefore, we try to fix that and get those things here..</li>
     * <li>No items chosen? Well, choose some new items. This might be anything
     * useful; a vial, a weapon we do not have yet, or maybe a thrown weapon.
     * At first, do not bother to forage items, which are not useful.</li>
     * <li>Still no item chosen? Then try to choose from all items, event those,
     * which are not useful right now. In other words, try to be more greedy.
     * This is to widen the navigation choices across the map. Remember that
     * the primary task of the wander drive is to search for enemies.</li>
     * <li>Still no items chosen? Instagib or no-weapons mutator removed all
     * items? What about running randomly along navigation points then?</li>
     * </ul>
     *
     * @return True, if the drive decided what shall we do now. False otherwise.
     */
    @Override public boolean doLogic ()
    {
        if (!main.optionWander)
            return false;

        // include wander logs?
        enableLogs (main.optionWanderLogs);

        // do we have an item to travel to?
        if (travelTicket > 0)
        {
            // we have an item, travel to it..
            if (travel.keepTraveling (travelTicket))
                return true;

            // do not travel anymore..
            log.fine ("Wander.doLogic(): done traveling to last item");
            travelTicket = -1;
        }

        // forage power-up items..
        if (choosePowerUp ())
            return true;

        // forage something handy..
        if (chooseItems (false))
            return true;

        // forage anything pickable, greedy style..
        if (chooseItems (true))
            return true;

        // just keep doing something funny
        return rambleThroughMap ();
    }

    /*========================================================================*/

    /**
     * Checks, whether necessary, then looks around and chooses some power-up
     * items: Health and armor, udamage. Initiates new travel ticket to the
     * nearest of them.
     *
     * @return True, if an action was taken. False otherwise.
     */
    protected boolean choosePowerUp ()
    {
        // do we want to power-up?
        if (
            (memory.self.getHealth () < 150)
            || (memory.self.getArmor () < 50)
        )
        {
            // retreive the list of possible items
            ArrayList<Item> items = getListOfGoodPowerup ();
            // travel to them
            return initTravel (items, "forage power-up items");
        }
        return false;
    }

    /**
     * Looks around and chooses some items.
     * Initiates new travel ticket to the nearest of them.
     *
     * @param greedy Whether to forage items we do not really need.
     * @return True, if an action was taken. False otherwise.
     */
    protected boolean chooseItems (boolean greedy)
    {
        // retreive the list of possible items
        ArrayList<Item> items = getListOfGoodItems (greedy);

        // do we have enough of items?
        if ( !greedy && (items.size () < 3) )
            return false;

        // travel to them
        if (initTravel (items, "forage items, greedy " + greedy))
            return true;

        log.finest (
            "Wander.chooseItems(): could not forage items"
            + ", count " + items.size()
        );

        return false;
    }

    /*========================================================================*/

    /**
     * Initiates new travel ticket to one of the given items.
     * @param items Items to be traveled to.
     * @param msg Log message for this travel.
     * @return Retruns true, if travel initialized.
     */
    private boolean initTravel (ArrayList<Item> items, String msg)
    {
        // item pickups are reported slowly, therefore remove items we're
        // standing on, since we won't be able to travel to them anyway..
        Iterator<Item> i = items.iterator ();
        while (i.hasNext ())
        {
            // is it close enough to the agent?
            if (memory.self.getSpaceDistance(i.next ().location) < LoqueNavigator.CLOSE_ENOUGH)
                // then do not add this item..
                i.remove ();
        }

        // no items to run to?
        if (items.size () <= 0)
            return false;

        // log decision info
        log.info ("Wander.initTravel(): " + msg + ", count " + items.size());
        // log chosen items..
        for (Item item : items)
        {
            log.finest (
                "Wander.initTravel(): " + item.cls
                + ", ID " + item.ID
                + ", " + item.UnrealID
            );
        }

        // we're going to travel to all of them
        travelTicket = travel.initTicketToAll (items);

        // are we traveling somewhere?
        return (travelTicket > 0);
    }

    /*========================================================================*/

    /**
     * Wanders mindlessly around the map, navpoint by navpoint. This method is
     * called, when there are no items on the map to be collected. Usually
     * because of Instagib or No-weapons mutator.
     *
     * <h4>Future</h4>
     *
     * This method should contain some kind of random rambling through nearby
     * navigation points in the high hope of finding something to live for.
     * Wandering arund items is not always working, since mutators can remove
     * a large portion of them.
     *
     * <p>There is probably no point in traveling somewhere far far away using
     * the tuned travel API. On the contrary, we should pick some nice nearby
     * navigation point and try to get to it. Jump, crouch, strafe, then double
     * jump and spin around. Make some fun of it! Well, maybe later.. ;)</p>
     *
     * @return True, if an action was taken. False otherwise.
     */
    protected boolean rambleThroughMap ()
    {
        return false;
    }

    /*========================================================================*/

    /** Loque pursue. */
    protected LoqueTravel travel;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     * @param travel Loque travel.
     */
    public LoqueWander (Main main, LoqueMemory memory, LoqueTravel travel)
    {
        super (main, memory);
        this.travel = travel;
    }
}